<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Cli;

use Exception;
use WP_CLI;
use WP_CLI_Command;

/**
 * CLI commands
 */
class Commands extends WP_CLI_Command {
	/**
	 * Sync advices.
	 *
	 * @param array $args args.
	 * @param array $assoc_args args.
	 *
	 * @return void
	 * @throws Exception WP cli.
	 */
	public function sync( $args = array(), $assoc_args = array() ) {
		$url    = array_key_exists( 'cpanel_url', $assoc_args ) ? $assoc_args['cpanel_url'] : '';
		$emails = array_key_exists( 'cpanel_user_emails', $assoc_args ) ? $assoc_args['cpanel_user_emails'] : '';

		do_action( 'cl_smart_advice_set_cpanel_url', $url );
		do_action( 'cl_smart_advice_set_cpanel_user_emails', $emails );

		$list = array_key_exists( 'list', $assoc_args ) ? $assoc_args['list'] : '';
		$data = json_decode( $list, true );

		if ( ! is_array( $data ) ) {
			// @phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
			do_action( 'cl_smart_advice_set_error', E_WARNING, 'Json ' . print_r( $list, true ) . ' is invalid.', __FILE__, __LINE__ );
			WP_CLI::error( 'Json is invalid.' );
		}

		do_action( 'cl_smart_advice_sync', $data );

		$failed = apply_filters( 'cl_smart_advice_sync_failed', false );

		if ( $failed ) {
			do_action( 'cl_smart_advice_set_error', E_WARNING, 'Failed sync advices.', __FILE__, __LINE__ );
			WP_CLI::error( 'Failed sync advices.' );
		}

		WP_CLI::success( 'Ok' );
	}

	/**
	 * Apply advice.
	 *
	 * @param array $args args.
	 * @param array $assoc_args args.
	 *
	 * @return void
	 * @throws Exception WP cli.
	 */
	public function apply( $args = array(), $assoc_args = array() ) {
		$advice_id   = $this->adviceId( $assoc_args );
		$advice_type = $this->adviceType( $assoc_args );
		$advice_uid  = $advice_type . '_' . $advice_id;
		do_action( 'cl_smart_advice_apply', $advice_uid, true );
		WP_CLI::success( 'Ok' );
	}

	/**
	 * Rollback advice.
	 *
	 * @param array $args args.
	 * @param array $assoc_args args.
	 *
	 * @return void
	 * @throws Exception WP cli.
	 */
	public function rollback( $args = array(), $assoc_args = array() ) {
		$advice_id   = $this->adviceId( $assoc_args );
		$advice_type = $this->adviceType( $assoc_args );
		$advice_uid  = $advice_type . '_' . $advice_id;
		$reason      = $this->reason( $assoc_args );
		do_action( 'cl_smart_advice_rollback', $advice_uid, true, $reason );
		WP_CLI::success( 'Ok' );
	}

	/**
	 * Delete self.
	 *
	 * @return void
	 */
	public function uninstall() {
		do_action( 'cl_smart_advice_uninstall', true );
	}

	/**
	 * Check advice id.
	 *
	 * @param array $assoc_args args.
	 *
	 * @return string
	 * @throws Exception WP cli.
	 */
	private function adviceId( $assoc_args ) {
		$advice_id = array_key_exists( 'advice_id', $assoc_args ) ? (string) $assoc_args['advice_id'] : '';
		if ( empty( $advice_id ) ) {
			WP_CLI::error( 'Advice id is empty' );
		}

		return $advice_id;
	}

	/**
	 * Check advice type.
	 *
	 * @param array $assoc_args args.
	 *
	 * @return string
	 * @throws Exception WP cli.
	 */
	private function adviceType( $assoc_args ) {
		$white_list  = array( 'awp', 'imunify' );
		$advice_type = array_key_exists( 'advice_type', $assoc_args ) ? (string) $assoc_args['advice_type'] : 'awp';
		if ( empty( $advice_type ) ) {
			WP_CLI::error( 'Advice type is empty' );
		} elseif ( ! in_array( $advice_type, $white_list ) ) {
			WP_CLI::error( 'Advice type is not valid, possible values: ' . implode( ', ', $white_list ) );
		}

		return $advice_type;
	}

	/**
	 * Sanitize rollback reason.
	 *
	 * @param array $assoc_args args.
	 *
	 * @return string
	 */
	private function reason( $assoc_args ) {
		return array_key_exists( 'reason', $assoc_args ) ? sanitize_text_field( wp_unslash( $assoc_args['reason'] ) ) : '';
	}
}
