<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Notice;

use CloudLinux\SmartAdvice\App\Option;

/**
 * Admin
 */
class Manager {

	/**
	 * Option.
	 *
	 * @var Option
	 */
	private $option;

	/**
	 * Constructor.
	 *
	 * @param Option $option manager.
	 */
	public function __construct( Option $option ) {
		$this->option = $option;

		add_action( 'wp_ajax_cl_smart_advice_notice_close', array( $this, 'ajaxClose' ) );
		add_action( 'cl_smart_advice_notice_add', array( $this, 'add' ), 10, 3 );
	}

	/**
	 * All notices.
	 *
	 * @return array
	 */
	public function all() {
		return $this->option->get( 'notices', true );
	}

	/**
	 * Get notice.
	 *
	 * @param string $notice_id id.
	 *
	 * @return ?Model
	 */
	public function get( $notice_id ) {
		$notices = $this->all();

		foreach ( $notices as $notice ) {
			if ( (string) $notice->id === (string) $notice_id ) {
				return $notice;
			}
		}

		return null;
	}

	/**
	 * Add notice.
	 *
	 * @param string $id notice.
	 * @param string $type notice.
	 * @param array  $payload data.
	 *
	 * @return void
	 */
	public function add( $id, $type, $payload ) {
		$notices = $this->all();

		$model = ( new Model() )->fill(
			array(
				'id'      => $id,
				'type'    => $type,
				'payload' => $payload,
			)
		);

		foreach ( $notices as $key => $notice ) {
			if ( (string) $notice->id === (string) $id ) {
				$notices[ $key ] = $model;

				$model = null;
			}
		}

		if ( ! empty( $model ) ) {
			$notices[] = $model;
		}

		$this->option->save( 'notices', $notices );
	}

	/**
	 * Display notices.
	 *
	 * @return void
	 */
	public function display() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$notices = $this->all();
		foreach ( $notices as $notice ) {

			if ( empty( $notice->info() ) ) {
				$this->delete( $notice->id );
				continue;
			}

			// phpcs:ignore
			echo $this->render( $notice );
		}
	}

	/**
	 * Render notice.
	 *
	 * @param Model $notice model.
	 *
	 * @return string
	 */
	public function render( Model $notice ) {
		$payload  = $notice->payload();
		$adv_list = '';
		if ( is_array( $payload ) && isset( $payload['list'] ) && ! empty( $payload['list'] ) ) {
			$adv_list = $payload['list'];
		}
		$html = '<div class="notice notice-' . esc_attr( $notice->htmlClass() ) . ' is-dismissible" data-cl-smart-advice-notice="' . esc_attr( $notice->id ) . '"';
		if ( ! empty( $adv_list ) ) {
			$html .= ' data-cl-smart-advice-list="' . esc_attr( $adv_list ) . '"';
		}
		$html .= '>';
		$html .= '<p>';
		$html .= '<strong>' . esc_attr( $notice->title() ) . '</strong><br>';
		$html .= $notice->htmlMessage();
		$html .= '</p>';
		$html .= '<button type="button" class="notice-dismiss" data-cl-smart-advice-notice-close="dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button>';
		$html .= '</div>';

		return $html;
	}

	/**
	 * Notice close sync.
	 *
	 * @param string $notice_id id.
	 *
	 * @return void
	 */
	public function delete( $notice_id ) {
		$notices = $this->all();

		foreach ( $notices as $key => $notice ) {
			if ( (string) $notice->id === (string) $notice_id ) {
				unset( $notices[ $key ] );
			}
		}

		$this->option->save( 'notices', $notices );
	}

	/**
	 * Ajax request: close.
	 *
	 * @return void
	 */
	public function ajaxClose() {
		$notice_id = array_key_exists( 'notice_id', $_REQUEST ) ? sanitize_text_field( wp_unslash( $_REQUEST['notice_id'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$this->delete( $notice_id );

		echo wp_json_encode( array() );

		wp_die();
	}
}

