<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App;

use CloudLinux\SmartAdvice;

/**
 * Option
 */
class Option {

	/**
	 * Option name.
	 *
	 * @var string
	 */
	private $option_name = 'cl_smart_advice';

	/**
	 * Relations.
	 *
	 * @var string[]
	 */
	private $relations = array(
		'advices'       => \CloudLinux\SmartAdvice\App\Advice\Model::class,
		'notices'       => \CloudLinux\SmartAdvice\App\Notice\Model::class,
		'logs'          => \CloudLinux\SmartAdvice\App\Advice\LogModel::class,
		'subscriptions' => \CloudLinux\SmartAdvice\App\Subscription\Model::class,
	);

	/**
	 * Clear data type.
	 *
	 * @var array<string,mixed>
	 */
	private $clearable = array(
		'version'            => null,
		'sync_at'            => null,
		'cpanel_url'         => null,
		'cpanel_user_emails' => null,
		'advices'            => array(),
		'notices'            => array(),
	);

	/**
	 * Storage.
	 *
	 * @var array
	 */
	private $storage = array();

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'pre_update_option_cl_smart_advice', array( $this, 'filterPreUpdateOption' ), 10, 3 );
		add_filter( 'option_cl_smart_advice', array( $this, 'filterGetOption' ), 10, 2 );
		add_filter( 'default_option_cl_smart_advice', array( $this, 'filterDefaultOption' ), 10, 3 );
		add_action( 'cl_smart_advice_deleting', array( $this, 'clear' ), 10, 0 );
		add_action( 'cl_smart_advice_set_cpanel_url', array( $this, 'setCPanelUrl' ), 10, 1 );
		add_action( 'cl_smart_advice_set_cpanel_user_emails', array( $this, 'setCPanelUserEmails' ), 10, 1 );
		$this->configure();
	}

	/**
	 * Configure storage.
	 *
	 * @return void
	 */
	private function configure() {
		$option = get_option( $this->option_name );
		if ( is_array( $option ) ) {
			$this->storage = $option;
		}
	}

	/**
	 * Storage.
	 *
	 * @return array
	 */
	public function storage() {
		return $this->storage;
	}

	/**
	 * Validate data.
	 *
	 * @param array $value option.
	 *
	 * @return array
	 */
	protected function validate( $value ) {
		if ( ! is_array( $value ) ) {
			$value = array();
		}

		$validated = array(
			'sync_at'            => array_key_exists( 'sync_at', $value ) ? $value['sync_at'] : null,
			'version'            => array_key_exists( 'version', $value ) ? $value['version'] : null,
			'cpanel_url'         => array_key_exists( 'cpanel_url', $value ) ? $value['cpanel_url'] : null,
			'cpanel_user_emails' => array_key_exists( 'cpanel_user_emails', $value ) ? $value['cpanel_user_emails'] : null,
		);

		foreach ( $this->relations as $type => $class ) {
			$validated[ $type ] = $this->validateModel( $type, $class, $value );
		}

		return $validated;
	}

	/**
	 * Validate advices.
	 *
	 * @param string $key option.
	 * @param string $class model.
	 * @param array  $value option.
	 *
	 * @return array
	 */
	protected function validateModel( $key, $class, $value ) {
		$data = array();

		if ( array_key_exists( $key, $value ) && is_array( $value[ $key ] ) ) {
			$data = $value[ $key ];
		}

		$data = array_map(
			function ( $item ) use ( $class ) {
				if ( is_array( $item ) ) {
					$model = ( new $class() )->fill( $item );
				} elseif ( $item instanceof $class ) {
					$model = $item;
				} else {
					$model = ( new $class() )->fill( array() );
				}

				if ( $model->isValid() ) {
					return $model->toArray();
				}

				return array();
			},
			$data
		);

		return array_filter(
			$data,
			function ( $item ) {
				return ! empty( $item );
			}
		);
	}

	/**
	 * Filter pre update
	 *
	 * @param mixed  $value The new, unserialized option value.
	 * @param mixed  $old_value The old option value.
	 * @param string $option Option name.
	 *
	 * @return array
	 */
	public function filterPreUpdateOption( $value, $old_value, $option ) {
		return $this->validate( $value );
	}

	/**
	 * Filter get option
	 *
	 * @param mixed  $value Value of the option. If stored serialized, it will be unserialized prior to being returned.
	 * @param string $option Option name.
	 *
	 * @return array
	 */
	public function filterGetOption( $value, $option ) {
		return $this->validate( $value );
	}

	/**
	 * Filters default option
	 *
	 * @param mixed  $default The default value to return if the option does not exist in the database.
	 * @param string $option Option name.
	 * @param bool   $passed_default Was `get_option()` passed a default value?.
	 *
	 * @return array
	 */
	public function filterDefaultOption( $default, $option, $passed_default ) {
		return $this->filterGetOption( $default, $option );
	}

	/**
	 * All.
	 *
	 * @return array
	 */
	public function all() {
		return $this->storage();
	}

	/**
	 * Get by key.
	 *
	 * @param string $key option.
	 * @param bool   $as_model instance.
	 *
	 * @return mixed|null
	 */
	public function get( $key, $as_model = false ) {
		$all = $this->all();

		if ( array_key_exists( $key, $all ) ) {
			$items = $all[ $key ];

			if ( $as_model ) {
				$class = $this->relations[ $key ];

				$items = array_map(
					function ( $item ) use ( $class ) {
						return ( new $class() )->fill( $item );
					},
					$items
				);

				$keys = array_map(
					function ( $item ) {
						return $item->uid();
					},
					$items
				);

				$items = array_combine( $keys, array_values( $items ) );
			}

			return $items;
		}

		return null;
	}

	/**
	 * Save data for key.
	 *
	 * @param string $key option.
	 * @param mixed  $data option.
	 *
	 * @return void
	 */
	public function save( $key, $data ) {
		$all         = $this->all();
		$all[ $key ] = $data;

		$this->storage = $this->validate( $all );
		update_option( $this->option_name, $this->storage(), false );
	}

	/**
	 * Set control panel url.
	 *
	 * @param string $url panel.
	 *
	 * @return void
	 */
	public function setCPanelUrl( $url ) {
		$url = strtolower( $url );
		$this->save( 'cpanel_url', $url );
	}

	/**
	 * Set control panel emails.
	 *
	 * @param string $emails panel.
	 *
	 * @return void
	 */
	public function setCPanelUserEmails( $emails ) {
		$this->save( 'cpanel_user_emails', $emails );
	}

	/**
	 * Panel url.
	 *
	 * @param bool $is_imunify url.
	 *
	 * @return string|null
	 */
	public function panelUrl( $is_imunify = false ) {
		$url = $this->get( 'cpanel_url' );
		if ( ! empty( $url ) ) {
			$parts = explode( '|', $url );

			// Backwards compatible for old version cllib < 3.3.7-2, since 0.6-9.
			if ( 2 === count( $parts ) ) {
				$type      = strtolower( $parts[0] );
				$panel_url = rtrim( $parts[1], '/' );

				if ( 'cpanel' === $type ) {
					$url = $panel_url . '/cpsess0000000000/frontend/paper_lantern/lveversion/wpos.live.pl';
				}

				if ( 'plesk' === $type ) {
					$url = $panel_url . '/modules/plesk-lvemanager/index.php/awp/index#/';
				}
			}

			// Fix cPanel url for Imunify.
			if ( $is_imunify && false !== strpos( $url, 'cpsess0000000000' ) ) {
				$parts = explode( '/lveversion', $url );
				$url   = $parts[0] . '/imunify/imunify.live.pl#/360/client/malware/files';
			}

			return $url;
		}

		return null;
	}

	/**
	 * Panel emails
	 *
	 * @return string|null
	 */
	public function panelEmails() {
		return $this->get( 'cpanel_user_emails' );
	}

	/**
	 * Delete smart advice related data.
	 *
	 * @return void
	 */
	public function clear() {
		$all = $this->all();

		foreach ( $this->clearable as $key => $type ) {
			if ( is_null( $type ) ) {
				unset( $all[ $key ] );
			} else {
				$all[ $key ] = $type;
			}
		}

		$this->storage = $all;
		update_option( $this->option_name, $this->storage(), false );
	}
}
