<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App;

use CloudLinux\SmartAdvice\App\Advice\Api;
use CloudLinux\SmartAdvice\App\Advice\ApiSocket;
use CloudLinux\SmartAdvice\App\Advice\Manager as AdviceManager;
use CloudLinux\SmartAdvice\App\Notice\Manager as NoticeManager;
use CloudLinux\SmartAdvice\App\Subscription\SubscriptionManager;
use CloudLinux\SmartAdvice\App\Service\Mail as MailService;
use CloudLinux\SmartAdvice\App\Service\Analytics;
use CloudLinux\SmartAdvice\App\Views\Admin;
use CloudLinux\SmartAdvice\App\Views\Settings;
use CloudLinux\SmartAdvice\App\Views\Widget;

/**
 * Initial class
 */
class Plugin {
	/**
	 * Self instance
	 *
	 * @var ?self
	 */
	private static $instance = null;

	/**
	 * Container.
	 *
	 * @var array
	 */
	private $container = array();

	/**
	 * Private constructor
	 */
	private function __construct() {
	}

	/**
	 * Private clone
	 */
	private function __clone() {
	}

	/**
	 * Get instance
	 *
	 * @return self
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Get service
	 *
	 * @param string $key class.
	 *
	 * @return mixed
	 */
	public function get( $key ) {
		if ( array_key_exists( $key, $this->container ) ) {
			return $this->container[ $key ];
		}

		return null;
	}

	/**
	 * Get environment
	 *
	 * @return string
	 */
	public function environment() {
		$environment = 'production';
		if (
			$this->isStagingModeDefined()
			||
			$this->isStagingFileExists()
		) {
			$environment = 'development';
		}

		return $environment;
	}

	/**
	 * Check CL_STAGING_MODE constant.
	 *
	 * @return bool
	 */
	public function isStagingModeDefined() {
		return ( defined( 'CL_STAGING_MODE' ) && CL_STAGING_MODE );
	}

	/**
	 * Check staging file.
	 *
	 * @return bool
	 */
	public function isStagingFileExists() {
		return @file_exists( '/opt/cloudlinux/staging_mode' ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
	}

	/**
	 * Setup container.
	 *
	 * @return void
	 */
	private function coreSetup() {
		$this->container[ Debug::class ]  = new Debug( $this->environment() );
		$this->container[ Api::class ]    = new ApiSocket();
		$this->container[ Option::class ] = new Option();
		$this->container[ Update::class ] = new Update( $this->get( Option::class ) );

		$this->get( Update::class )->update();

		$this->container[ Analytics::class ] = new Analytics( $this->environment() );

		$this->container[ NoticeManager::class ] = new NoticeManager( $this->get( Option::class ) );
		$this->container[ AdviceManager::class ] = new AdviceManager( $this->get( Api::class ), $this->get( Option::class ) );

		$this->container[ SubscriptionManager::class ] = new SubscriptionManager(
			$this->get( Option::class ),
			$this->get( AdviceManager::class ),
			$this->get( Analytics::class )
		);

		$this->container[ MailService::class ] = new MailService(
			$this->get( AdviceManager::class )->domain(),
			$this->get( AdviceManager::class )->home(),
			$this->get( Option::class ),
			$this->get( Analytics::class )
		);

		add_action( 'shutdown', array( $this, 'shutdown' ) );
		add_action( 'cl_smart_advice_uninstall', array( $this, 'shutdown' ) );
	}

	/**
	 * Additional setup for WP Admin env.
	 *
	 * @return void
	 */
	private function adminSetup() {
		$this->container[ Admin::class ]    = new Admin( $this->get( Analytics::class ), $this->get( Debug::class ) );
		$this->container[ Widget::class ]   = new Widget( $this->get( AdviceManager::class ) );
		$this->container[ Settings::class ] = new Settings( $this->get( AdviceManager::class ), $this->get( NoticeManager::class ) );
	}

	/**
	 * Init plugin.
	 *
	 * @return void
	 */
	public function init() {
		$this->coreSetup();

		if ( is_admin() ) {
			$this->adminSetup();
		}
	}

	/**
	 * Auto delete.
	 *
	 * @param bool $do_uninstall Delete plugin symlink.
	 *
	 * @return void
	 */
	public function shutdown( $do_uninstall = false ) {
		if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
			return;
		}

		/* @var AdviceManager $manager advices. */
		$manager = $this->get( AdviceManager::class );

		$sync_failed = apply_filters( 'cl_smart_advice_sync_failed', false );

		if (
			( empty( $manager->syncAt() ) && $sync_failed ) ||
			( ! empty( $manager->syncAt() ) && empty( $manager->advices() ) && ! $sync_failed ) ||
			true === $do_uninstall
		) {
			do_action( 'cl_smart_advice_deleting' );
		}
	}
}
