<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Service;

use CloudLinux\SmartAdvice\App\Advice\Model;
use CloudLinux\SmartAdvice\App\Option;

/**
 * Mail advices
 */
class MailerAdvices extends Mailer {
	/**
	 * Option.
	 *
	 * @var Option
	 */
	protected $option;

	/**
	 * Analytics service.
	 *
	 * @var Analytics
	 */
	protected $analytics;

	/**
	 * Number of new advices.
	 *
	 * @var int
	 */
	protected $count = 0;

	/**
	 * Advice.
	 *
	 * @var Model
	 */
	protected $advice = null;

	/**
	 * Links list to track with analytics. [event name -> parameter name].
	 *
	 * @var array
	 */
	protected $links_to_track = array(
		'email_link_clicked'        => 'panel_link',
		'email_cancel_link_clicked' => 'unsubscribe_link',
	);

	/**
	 * Analytics events aliases. [event name -> event alias].
	 *
	 * @var array
	 */
	protected $analytic_events = array(
		'email_opened' => 'email_opened',
		'email_sent'   => 'email_sent',
		'email_error'  => 'email_error',
	);

	/**
	 * Constructor.
	 *
	 * @param Option    $option manager.
	 * @param Analytics $analytics analytics.
	 * @param array     $payload data.
	 */
	public function __construct( Option $option, Analytics $analytics, $payload ) {
		$this->option    = $option;
		$this->analytics = $analytics;
		$this->prepare_payload( $payload );
	}

	/**
	 * Prepare payload data.
	 *
	 * @param array $payload data.
	 *
	 * @return void
	 */
	protected function prepare_payload( $payload ) {
		if ( array_key_exists( 'advice', $payload ) && $payload['advice'] instanceof Model ) {
			$this->advice = $payload['advice'];
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public function allowed() {
		return ! is_null( $this->advice );
	}

	/**
	 * Get option.
	 *
	 * @return Option
	 */
	protected function option() {
		return $this->option;
	}

	/**
	 * Get analytics.
	 *
	 * @return Analytics
	 */
	protected function analytics() {
		return $this->analytics;
	}

	/**
	 * {@inheritDoc}
	 */
	public function subject( $site_url ) {

		// Advice object is definitely present at this point as we already checked in allowed() method.
		$subject = $this->advice()->get_email_subject( $site_url );
		if ( empty( $subject ) ) {
			if ( $this->advice()->is_imunify() ) {
				$subject = 'New security Advice';
			} else {
				$subject = 'New performance Advice';
			}
			if ( ! empty( $site_url ) ) {
				$subject .= ' for ' . $site_url;
			}
		}

		return $subject;
	}

	/**
	 * Template file.
	 *
	 * @return string
	 */
	public function template() {
		return 'letter.advices';
	}

	/**
	 * Advice.
	 *
	 * @return Model
	 */
	public function advice() {
		return $this->advice;
	}

	/**
	 * {@inheritDoc}
	 */
	public function filter_template_data( $data, $email, $user_type, $email_type ) {
		if ( ! is_array( $data ) ) {
			return $data;
		}

		$data = array_merge(
			$data,
			array(
				'advice_title'            => $this->advice()->description,
				'advice_description'      => $this->advice()->detailed_description,
				'view_advice_button_text' => $this->advice()->get_email_view_advice_text(),
			)
		);

		/**
		 * Filter the data used for the email template.
		 *
		 * @param array  $data The data to use in template.
		 * @param string $email Email address.
		 * @param string $user_type User type.
		 * @param string $advice_type Advice type.
		 * @param string $email_type Email type.
		 *
		 * @return array
		 * @since  0.1-6
		 */
		$data = apply_filters( 'cl_smart_advice_email_template_data', $data, $email, $user_type, $this->advice()->type, $email_type );

		// Replace the original links to advice UI with redirect via Smart Advice analytics service.
		if ( $this->advice()->is_analytics_available() ) {
			foreach ( $this->links_to_track as $event => $link ) {
				if ( array_key_exists( $link, $data ) ) {
					$data[ $link ] = $this->analytics()->get_redirect_link(
						$email,
						$this->advice()->id,
						$event,
						$data[ $link ]
					);
				}
			}

			// Add analytics pixel to the footer.
			$footer         = array_key_exists( 'footer', $data ) ? $data['footer'] : '';
			$data['footer'] = $footer . sprintf(
				'<img src="%s" width="1" height="1" />',
				$this->analytics()->get_pixel_url( $email, $this->advice()->id, $this->analytic_events['email_opened'] )
			);
		}

		return $data;
	}

	/**
	 * {@inheritDoc}
	 */
	public function post_email_sent( $success, $user_type, $email, $subject, $body, $headers ) {
		if ( $this->advice()->is_analytics_available() ) {
			$this->analytics()->send_event(
				$email,
				$this->advice()->id,
				$success ? $this->analytic_events['email_sent'] : $this->analytic_events['email_error']
			);
		}
	}
}
