<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App;

use CloudLinux\SmartAdvice;
use CloudLinux\SmartAdvice\App\Notice\Model as NoticeModel;
use CloudLinux\SmartAdvice\App\Advice\LogModel as LogModel;

/**
 * Option
 */
class Update {

	/**
	 * Option model instance.
	 *
	 * @var Option instance.
	 */
	protected $option;

	/**
	 * Constructor.
	 *
	 * @param Option $option instance.
	 */
	public function __construct( Option $option ) {
		$this->option = $option;
	}

	/**
	 * Get option instance.
	 *
	 * @return Option
	 */
	public function option() {
		return $this->option;
	}

	/**
	 * Get version of plugin.
	 *
	 * @return string|null
	 */
	public function version() {
		return $this->option()->get( 'version' );
	}

	/**
	 * Get current version of plugin.
	 *
	 * @return string
	 */
	public function currentVersion() {
		return CL_SMART_ADVICE_VERSION;
	}

	/**
	 * Update if needed.
	 *
	 * @return void
	 */
	public function update() {
		$version = $this->version();

		// Lock update.
		if ( $this->currentVersion() !== $version ) {
			$this->option()->save( 'version', CL_SMART_ADVICE_VERSION );
		}

		if ( empty( $version ) || $this->currentVersion() === 'dev' ) {
			return;
		}

		if ( version_compare( $version, '0.6-25', '<' ) ) {
			$this->update_06_25();
		}
	}

	/**
	 * Update to 0.6-25 version.
	 *
	 * @return void
	 */
	protected function update_06_25() {
		// Update notices model. Before this update we store advice_id. Not it must be advice uid.

		$notices = $this->option()->get( 'notices', true );
		foreach ( $notices as $key => $notice ) {
			if ( is_numeric( $notice->id ) ) {
				$notice->id = 'awp_' . $notice->id;
			}

			if ( is_array( $notice->payload ) ) {
				if (
					array_key_exists( 'list', $notice->payload ) &&
					is_string( $notice->payload['list'] )
				) {
					$list = explode( ',', $notice->payload['list'] );
					foreach ( $list as $list_key => $advice_id ) {
						$list[ $list_key ] = 'awp_' . $advice_id;
					}
					$notice->payload['list'] = implode( ',', $list );
				}

				if ( array_key_exists( 'advice_id', $notice->payload ) ) {
					$notice->payload['advice_uid'] = 'awp_' . $notice->payload['advice_id'];
					unset( $notice->payload['advice_id'] );
				}
			}

			unset( $notices[ $key ] );

			$notices[ $notice->id ] = $notice;
		}
		$this->option()->save( 'notices', $notices );

		// Update logs id with advice uid.
		$logs = $this->option()->get( 'logs', true );
		foreach ( $logs as $key => $log ) {
			$log->id = 'awp_' . $log->id;
			unset( $logs[ $key ] );
			$logs[ $log->id ] = $log;
		}
		$this->option()->save( 'logs', $logs );
	}
}
