<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Advice;

use CloudLinux\SmartAdvice\App\Model as BaseModel;

/**
 * Advice model
 */
class Model extends BaseModel {
	/**
	 * Advice ID.
	 *
	 * @var string
	 */
	public $id = '';

	/**
	 * Advice type.
	 *
	 * @var string
	 */
	public $type = '';

	/**
	 * Advice status.
	 *
	 * @var string
	 */
	public $status = '';

	/**
	 * Advice short description.
	 *
	 * @var string
	 */
	public $description = '';

	/**
	 * Advice detail description.
	 *
	 * @var string
	 */
	public $detailed_description = '';

	/**
	 * CTA text for the "Apply advice" button.
	 *
	 * @var string
	 */
	public $apply_advice_button_text = '';

	/**
	 * CTA text for the "Upgrade to apply" button.
	 *
	 * @var string
	 */
	public $upgrade_to_apply_button_text = '';

	/**
	 * CTA email text for the "Show advice" button in email.
	 *
	 * @var string
	 */
	public $email_view_advice_text = '';

	/**
	 * The subject of the email letter with 1 possible substitution - website URL ({url})
	 *
	 * @var string
	 */
	public $email_subject = '';

	/**
	 * Advice note.
	 *
	 * @var string
	 */
	public $note = '';

	/**
	 * Is premium.
	 *
	 * @var bool
	 */
	public $is_premium = false;

	/**
	 * Module name.
	 *
	 * @var string
	 */
	public $module_name = '';

	/**
	 * License status.
	 *
	 * @var string
	 */
	public $license_status = '';

	/**
	 * Requests.
	 *
	 * @var array
	 */
	public $requests = array();

	/**
	 * Advice total stages.
	 *
	 * @var int
	 */
	public $total_stages = 0;

	/**
	 * Advice completed stages.
	 *
	 * @var int
	 */
	public $completed_stages = 0;

	/**
	 * Subscription status.
	 * no = need subscription.
	 *
	 * @var string
	 */
	public $subscription_status = '';

	/**
	 * Subscription upgrade url.
	 *
	 * @var string
	 */
	public $subscription_upgrade_url = '';

	/**
	 * Username.
	 *
	 * @var string
	 */
	public $username = '';

	/**
	 * Advice created at.
	 *
	 * @var string
	 */
	public $created_at = '';

	/**
	 * Advice updated at.
	 *
	 * @var string
	 */
	public $updated_at = '';

	/**
	 * Fill properties.
	 *
	 * @param array $data data.
	 *
	 * @return self
	 */
	public function fill( $data ) {
		if ( array_key_exists( 'subscription', $data ) ) {
			if ( array_key_exists( 'status', $data['subscription'] ) ) {
				$data['subscription_status'] = $data['subscription']['status'];
			}
			if ( array_key_exists( 'upgrade_url', $data['subscription'] ) ) {
				$data['subscription_upgrade_url'] = $data['subscription']['upgrade_url'];
			}
		}

		parent::fill( $data );

		return $this;
	}

	/**
	 * Is valid model.
	 *
	 * @return bool
	 */
	public function isValid() {
		return ! empty( $this->id ) && ! empty( $this->type ) && ! empty( $this->status );
	}

	/**
	 * To array.
	 *
	 * @return array
	 */
	public function toArray() {
		return array(
			'id'                           => $this->id,
			'uid'                          => $this->uid(),
			'type'                         => $this->type,
			'status'                       => $this->status,
			'description'                  => $this->description,
			'detailed_description'         => $this->detailed_description,
			'apply_advice_button_text'     => $this->apply_advice_button_text,
			'upgrade_to_apply_button_text' => $this->upgrade_to_apply_button_text,
			'email_view_advice_text'       => $this->email_view_advice_text,
			'email_subject'                => $this->email_subject,
			'note'                         => $this->note,
			'is_premium'                   => $this->is_premium,
			'module_name'                  => $this->module_name,
			'license_status'               => $this->license_status,
			'requests'                     => $this->requests,
			'total_stages'                 => $this->total_stages,
			'completed_stages'             => $this->completed_stages,
			'subscription_status'          => $this->subscription_status,
			'subscription_upgrade_url'     => $this->subscription_upgrade_url,
			'username'                     => $this->username,
			'created_at'                   => $this->created_at,
			'updated_at'                   => $this->updated_at,
		);
	}

	/**
	 * Returns a unique identifier for sharing advice between AccelerateWp/Imunify modules
	 *
	 * @return string
	 */
	public function uid() {
		if ( $this->is_imunify() ) {
			return 'imunify_' . $this->id;
		}

		return 'awp_' . $this->id;
	}

	/**
	 * Get stage percent.
	 *
	 * @return float
	 */
	public function stagePercent() {
		$total     = $this->total_stages;
		$completed = $this->completed_stages;

		if ( $total > 0 ) {
			return round( $completed / $total * 100 );
		}

		return 0;
	}

	/**
	 * Get spinner percent.
	 *
	 * @return string
	 */
	public function spinnerPercent() {
		$percent = $this->stagePercent();
		if ( $percent > 0 ) {
			return round( $percent / 10 ) . '0';
		}

		return '0';
	}

	/**
	 * Status for ui logic.
	 *
	 * @return string
	 */
	public function statusUi() {
		$status = $this->status;

		if ( 'pending' === $this->subscription_status ) {
			$status = 'pending';
		}

		return $status;
	}

	/**
	 * Need subscription.
	 *
	 * @return string
	 */
	public function subscriptionUpgradeUrl() {
		return (string) $this->subscription_upgrade_url;
	}

	/**
	 * Need agreement.
	 *
	 * @return false|string
	 */
	public function agreementType() {
		$statuses = array(
			'NOT_REQUIRED'    => false,
			'NOT_APPROVED'    => true,
			'UPDATE_REQUIRED' => true,
			'APPROVED'        => false,
		);

		if ( array_key_exists( $this->license_status, $statuses ) && true === $statuses[ $this->license_status ] ) {
			if ( 'cdn' === strtolower( $this->type ) ) {
				return 'cdn';
			}
		}

		return false;
	}

	/**
	 * Determines the CTA text for the "Apply advice" button.
	 *
	 * @return string
	 * @since 0.1-8
	 */
	public function get_apply_advice_button_text() {
		return empty( $this->apply_advice_button_text ) ? esc_html__( 'Apply', 'cl-smart-advice' ) : $this->apply_advice_button_text;
	}

	/**
	 * Determines the CTA text for the "Upgrade to apply" button.
	 *
	 * @return string
	 * @since 0.1-8
	 */
	public function get_upgrade_to_apply_button_text() {
		if ( $this->is_imunify() ) {
			return esc_html__( 'Buy now', 'cl-smart-advice' );
		}

		return empty( $this->upgrade_to_apply_button_text ) ? esc_html__( 'Upgrade to apply', 'cl-smart-advice' ) : $this->upgrade_to_apply_button_text;
	}


	/**
	 * Determines the CTA email text for the "Show advice" button in email.
	 *
	 * @return string
	 * @since 0.1-8
	 */
	public function get_email_view_advice_text() {
		return empty( $this->email_view_advice_text ) ? esc_html__( 'View advice', 'cl-smart-advice' ) : $this->email_view_advice_text;
	}

	/**
	 * Determines the subject of the email letter with 1 possible substitution - website URL ({url}).
	 *
	 * @param string $site_url Site URL.
	 *
	 * @return string
	 *
	 * @since 0.1-8
	 */
	public function get_email_subject( $site_url ) {

		if ( empty( $this->email_subject ) ) {
			// Default email subject is defined by the mailer.
			return '';
		}

		return preg_replace( '/{url}/', $site_url, $this->email_subject );
	}

	/**
	 * Is Imunify advice.
	 *
	 * @return bool
	 */
	public function is_imunify() {
		return false !== strpos( strtolower( $this->type ), 'imunify' );
	}

	/**
	 * Send email notifications on this advice.
	 *
	 * @return bool
	 */
	public function is_email_available() {
		if ( 'review' !== $this->status ) {
			return false;
		}

		return true;
	}

	/**
	 * Is analytics available for this advice.
	 *
	 * @return bool
	 */
	public function is_analytics_available() {
		return $this->is_imunify() === false;
	}
}
