<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Notice;

use CloudLinux\SmartAdvice\App\Model as BaseModel;

/**
 * Advice model
 */
class Model extends BaseModel {
	const NEW_ADVICES = array(
		'is_error' => false,
		'type'     => 'NEW_ADVICES',
		'class'    => 'success',
		'message'  => 'You have new Advice (#count#) to improve the performance of your site.' . PHP_EOL . '<a href="#url#" class="cl-smart-advice__notice-button" data-cl-smart-advice-notice-close="view">View advice</a>',
		'payload'  => array(),
	);

	const NEW_ADVICES_IMUNIFY = array(
		'is_error' => false,
		'type'     => 'NEW_ADVICES_IMUNIFY',
		'class'    => 'success',
		'message'  => 'You have new, urgent Advice (#count#) to cleanup malware and enable protection for your website.' . PHP_EOL . '<a href="#url#" class="cl-smart-advice__notice-button" data-cl-smart-advice-notice-close="view">View advice</a>',
		'payload'  => array(),
	);

	const ERROR_ADVICE_APPLY = array(
		'is_error' => true,
		'type'     => 'ERROR_ADVICE_APPLY',
		'class'    => 'error',
		'message'  => 'Error applying advice, please try again later.',
		'payload'  => array(),
	);

	const ERROR_ADVICE_ROLLBACK = array(
		'is_error' => true,
		'type'     => 'ERROR_ADVICE_ROLLBACK',
		'class'    => 'error',
		'message'  => 'Rollback advice error, please try again later.',
		'payload'  => array(),
	);

	const ERROR_ADVICE_STATUS = array(
		'is_error' => true,
		'type'     => 'ERROR_ADVICE_STATUS',
		'class'    => 'error',
		'message'  => '',
		'payload'  => array(),
	);

	const ERROR_ADVICE_NOT_FOUND = array(
		'is_error' => true,
		'type'     => 'ERROR_ADVICE_NOT_FOUND',
		'class'    => 'error',
		'message'  => 'Advice not found.',
		'payload'  => array(),
	);

	/**
	 * Advice ID.
	 *
	 * @var string
	 */
	public $id = '';

	/**
	 * Advice type.
	 *
	 * @var string
	 */
	public $type = '';

	/**
	 * Notice payload.
	 *
	 * @var array
	 */
	public $payload = array();

	/**
	 * Notice info.
	 *
	 * @var array
	 */
	public $info = array();

	/**
	 * Get uid.
	 *
	 * @return string
	 */
	public function uid() {
		return $this->id;
	}

	/**
	 * Is valid model.
	 *
	 * @return bool
	 */
	public function isValid() {
		return ! empty( $this->id ) && ! empty( $this->type ) && is_array( $this->payload );
	}

	/**
	 * To array.
	 *
	 * @return array
	 */
	public function toArray() {
		return array(
			'id'      => $this->id,
			'type'    => $this->type,
			'payload' => $this->payload,
		);
	}

	/**
	 * Payload.
	 *
	 * @return array
	 */
	public function payload() {
		$payload = $this->payload;
		if ( ! is_array( $payload ) ) {
			$payload = array();
		}

		return $payload;
	}

	/**
	 * Type info.
	 *
	 * @return string[]|array[]|null
	 */
	public function info() {
		if ( ! empty( $this->info ) ) {
			return $this->info;
		}

		$this->info = array();

		if ( defined( __CLASS__ . '::' . $this->type ) ) {
			$this->info = constant( __CLASS__ . '::' . $this->type );
		}

		return $this->info;
	}

	/**
	 * Html class.
	 *
	 * @return string
	 */
	public function htmlClass() {
		$info = $this->info();
		if ( array_key_exists( 'class', $info ) ) {
			return $info['class'];
		}

		return '';
	}

	/**
	 * Title.
	 *
	 * @return string
	 */
	public function title() {
		$title = 'SmartAdvice';

		$info = $this->info();

		if ( empty( $info ) ) {
			return $title;
		}

		$subtitle = '';
		$payload  = $this->payload();

		if ( array_key_exists( 'advice_type', $payload ) ) {
			$subtitle = $this->titleAdviceType( (string) $payload['advice_type'] );
		}

		if ( ! empty( $subtitle ) ) {
			$title .= ' - ' . $subtitle;
		}

		return $title;
	}

	/**
	 * Advice title.
	 *
	 * @param string $type of advice.
	 *
	 * @return string
	 */
	private function titleAdviceType( $type ) {
		$advice_type = strtolower( $type );
		switch ( $advice_type ) {
			case 'object_cache':
				return 'Object Caching';
			case 'site_optimization':
				return 'AccelerateWP';
			case 'cdn':
				return 'CDN';
			case 'cpcss':
				return 'Critical Path CSS';
			case 'image_optimization':
				return 'Image Optimization';
			default:
				return (string) str_replace( '_', ' ', $type );
		}
	}

	/**
	 * Html message.
	 *
	 * @return string
	 */
	public function htmlMessage() {
		$info = $this->info();

		if ( empty( $info ) ) {
			return '';
		}

		$payload        = $this->payload();
		$payload['url'] = esc_attr( admin_url( 'options-general.php?page=' . CL_SMART_ADVICE_SLUG ) );

		if ( array_key_exists( 'action', $info['payload'] ) ) {
			$payload['url'] .= '&action=' . $info['payload']['action'];
		}

		$keys = array_keys( $payload );
		$keys = array_map(
			function ( $key ) {
				return '#' . $key . '#';
			},
			$keys
		);

		$values = array_values( $payload );

		$message = str_replace( $keys, $values, $info['message'] );

		if ( array_key_exists( 'description', $payload ) ) {
			if ( ! empty( $message ) ) {
				$message .= PHP_EOL;
			}
			$message .= $this->payload['description'];
		}

		if ( function_exists( 'nl2br' ) ) {
			return nl2br( $message );
		}

		return $message;
	}

	/**
	 * Is error.
	 *
	 * @return bool
	 */
	public function isError() {
		$info = $this->info();
		if ( array_key_exists( 'is_error', $info ) ) {
			return (bool) $info['is_error'];
		}

		return false;
	}
}
