<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Service;

use CloudLinux\SmartAdvice\App\Advice\Model;
use CloudLinux\SmartAdvice\App\Option;
use WP_Error;

/**
 * Mail service
 */
class Mail {
	/**
	 * Panel user type string.
	 *
	 * @var string
	 */
	const USER_PANEL_TYPE = 'panel';

	/**
	 * WordPress' user type string.
	 *
	 * @var string
	 */
	const USER_WP_TYPE = 'wp';

	/**
	 * Domain.
	 *
	 * @var string
	 */
	private $domain;

	/**
	 * Home url.
	 *
	 * @var string
	 */
	private $home;

	/**
	 * Option.
	 *
	 * @var Option
	 */
	private $option;

	/**
	 * Analytics.
	 *
	 * @var Analytics
	 */
	private $analytics;

	/**
	 * Advice.
	 *
	 * @var ?Model
	 */
	protected $advice = null;

	/**
	 * Mailers.
	 *
	 * @var array
	 */
	private $mailers = array(
		'advices'   => MailerAdvices::class,
		'reminders' => MailerReminders::class,
	);

	/**
	 * Mailer.
	 *
	 * @var ?Mailer
	 */
	private $mailer = null;

	/**
	 * Constructor.
	 *
	 * @param string    $domain site.
	 * @param string    $home url.
	 * @param Option    $option manager.
	 * @param Analytics $analytics analytics.
	 */
	public function __construct( $domain, $home, $option, $analytics ) {
		$this->domain    = $domain;
		$this->home      = $home;
		$this->option    = $option;
		$this->analytics = $analytics;
		add_action( 'cl_smart_advice_email', array( $this, 'notify' ), 10, 2 );
		add_action( 'wp_mail_failed', array( $this, 'processMailError' ), 10, 1 );
	}

	/**
	 * Domain.
	 *
	 * @return string
	 */
	public function domain() {
		return $this->domain;
	}

	/**
	 * Home.
	 *
	 * @return string
	 */
	public function home() {
		return $this->home;
	}

	/**
	 * Option.
	 *
	 * @return Option
	 */
	public function option() {
		return $this->option;
	}

	/**
	 * Analytics.
	 *
	 * @return Analytics
	 */
	public function analytics() {
		return $this->analytics;
	}

	/**
	 * Advice.
	 *
	 * @return ?Model
	 */
	public function advice() {
		return $this->advice;
	}

	/**
	 * Is Imunify Advice.
	 *
	 * @return bool
	 */
	public function is_imunify() {
		return ( $this->advice() && $this->advice()->is_imunify() );
	}

	/**
	 * Prepare payload data.
	 *
	 * @param array $payload data.
	 *
	 * @return void
	 */
	protected function prepare_payload( $payload ) {
		$this->advice = null;
		if ( array_key_exists( 'advice', $payload ) && $payload['advice'] instanceof Model ) {
			$this->advice = $payload['advice'];
		}
	}

	/**
	 * Panel url.
	 *
	 * @return string
	 */
	public function panelUrl() {
		$url = $this->option()->panelUrl( $this->is_imunify() );

		return ! empty( $url ) ? $url : '';
	}

	/**
	 * Panel emails.
	 *
	 * @return string
	 */
	public function panelEmails() {
		$emails = $this->option()->panelEmails();

		return ! empty( $emails ) ? $emails : '';
	}

	/**
	 * Mailer.
	 *
	 * @return ?Mailer
	 */
	public function mailer() {
		return $this->mailer;
	}

	/**
	 * Set mailer.
	 *
	 * @param string $type Type of mailer.
	 * @param array  $payload data.
	 *
	 * @return void
	 */
	public function setMailer( $type, array $payload ) {
		$this->mailer = null;
		if ( array_key_exists( $type, $this->mailers ) ) {
			$this->mailer = new $this->mailers[ $type ]( $this->option(), $this->analytics(), $payload );
		}
	}

	/**
	 * Notifying.
	 *
	 * @param string $email_type mailer.
	 * @param array  $payload data.
	 *
	 * @return void
	 */
	public function notify( $email_type, $payload ) {
		do_action( 'cl_smart_advice_set_error_handler' );

		$this->prepare_payload( $payload );
		$this->setMailer( $email_type, $payload );

		if ( ! is_null( $this->mailer() ) && $this->mailer()->allowed() ) {
			$recipients = $this->recipients( $this->panelEmails() );
			$template   = $this->mailer()->template();
			$subject    = $this->mailer()->subject( $this->domain() );
			$headers    = $this->mailer()->headers();

			foreach ( $recipients as $user_type => $emails ) {
				$attr = $this->attr( $user_type, $payload );

				foreach ( $emails as $email ) {
					/**
					 * Allow other checks to be applied via filter.
					 *
					 * @param bool   $allowed Is reminder allowed?
					 * @param string $email_type Email type - first notification or reminder.
					 * @param string $email Email address.
					 * @param string $user_type User type.
					 * @param array  $payload Data.
					 *
					 * @return bool
					 * @since  0.1-6
					 */
					if ( ! apply_filters( 'cl_smart_advice_email_allowed', true, $email_type, $email, $user_type, $payload ) ) {
						continue;
					}

					$body = $this->body( $template, $this->mailer()->filter_template_data( $attr, $email, $user_type, $email_type ) );

					$send = $this->send( $user_type, $email, $subject, $body, $headers );

					$this->mailer()->post_email_sent( $send, $user_type, $email, $subject, $body, $headers );
				}
			}
		}

		do_action( 'cl_smart_advice_restore_error_handler' );
	}

	/**
	 * Merge WP users and Panel with duplicate filtering
	 *
	 * @param string $panel_user_emails List of emails.
	 *
	 * @return array
	 */
	public function recipients( $panel_user_emails ) {
		$admin_users  = get_users( array( 'role__in' => array( 'Administrator' ) ) );
		$admin_emails = array();
		foreach ( $admin_users as $user ) {
			if ( ! empty( $user->user_email ) ) {
				$admin_emails[] = $user->user_email;
			}
		}

		$panel_emails = explode( ',', $panel_user_emails );
		$panel_emails = array_map( 'trim', $panel_emails );
		$panel_emails = array_diff( $panel_emails, array( '' ) );
		$panel_emails = array_diff( $panel_emails, $admin_emails );

		return array(
			self::USER_PANEL_TYPE => $panel_emails,
			self::USER_WP_TYPE    => $admin_emails,
		);
	}

	/**
	 * Render body
	 *
	 * @param string $template template.
	 * @param array  $attr data.
	 *
	 * @return ?string
	 */
	public function body( $template, $attr ) {
		$path = CL_SMART_ADVICE_FOLDER_PATH . '/views/' . $template . '.php';
		if ( is_readable( $path ) ) {
			foreach ( $attr as $key => $val ) {
				unset( $attr[ $key ] );
				if ( is_array( $val ) ) {
					continue;
				}
				$attr[ '{{ ' . $key . ' }}' ] = $val;
			}

			return strtr( file_get_contents( $path ), $attr ) . '<!--CloudLinux-->'; // @phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
		}

		return null;
	}

	/**
	 * Formatted template attributes.
	 *
	 * @param string $user_type user type.
	 * @param array  $payload data.
	 *
	 * @return array
	 */
	public function attr( $user_type, $payload ) {

		$attr = array(
			'site_link'  => $this->home(),
			'site_url'   => $this->domain(),
			'panel_link' => esc_attr( admin_url( 'options-general.php?page=' . CL_SMART_ADVICE_SLUG ) ),
			'footer'     => '',
		);

		if ( self::USER_PANEL_TYPE === $user_type ) {
			$panel_url = $this->panelUrl();
			if ( ! empty( $panel_url ) ) {
				$attr['panel_link'] = $panel_url;
			}
		}

		return array_merge( $attr, array( 'payload' => $payload ) );
	}

	/**
	 * Send emails.
	 *
	 * @param string $user_type Type of user.
	 * @param string $email Email.
	 * @param string $subject Letter subject.
	 * @param string $body Letter body.
	 * @param array  $headers Letter header.
	 *
	 * @return bool
	 */
	public function send( $user_type, $email, $subject, $body, $headers ) {
		if ( self::USER_PANEL_TYPE === $user_type ) {
			add_filter( 'wp_mail_from', array( $this, 'panelSenderMail' ) );
			add_filter( 'wp_mail_from_name', array( $this, 'panelSenderName' ) );
		}

		$send = wp_mail( $email, $subject, $body, $headers );

		if ( self::USER_PANEL_TYPE === $user_type ) {
			remove_filter( 'wp_mail_from', array( $this, 'panelSenderMail' ) );
			remove_filter( 'wp_mail_from_name', array( $this, 'panelSenderName' ) );
		}

		return $send;
	}

	/**
	 * A cPanel Sender Mailbox.
	 *
	 * @return string
	 */
	public function panelSenderMail() {
		if ( $this->is_imunify() ) {
			return 'MyImunify@' . $this->domain();
		} else {
			return 'AccelerateWP@' . $this->domain();
		}
	}

	/**
	 * A cPanel Sender Name.
	 *
	 * @return string
	 */
	public function panelSenderName() {
		if ( $this->is_imunify() ) {
			return 'MyImunify';
		} else {
			return 'AccelerateWP';
		}
	}

	/**
	 * Send sentry event when email sending error.
	 *
	 * @param WP_Error $wp_error WP Error instance.
	 *
	 * @return void
	 */
	public function processMailError( $wp_error ) {
		/* @phpstan-ignore-next-line */
		if ( is_wp_error( $wp_error ) ) {
			$error_data = $wp_error->get_error_data();
			if (
				is_array( $error_data )
				&& isset( $error_data['message'] )
				&& false !== strpos( $error_data['message'], '<!--CloudLinux-->' )
			) {
				do_action(
					'cl_smart_advice_set_error',
					E_ERROR,
					'Email sending error: ' . $wp_error->get_error_message(),
					__FILE__,
					__LINE__,
					array(
						'to' => isset( $error_data['to'] ) ? $error_data['to'] : '',
					)
				);
			}
		}
	}
}
