<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Views;

use CloudLinux\SmartAdvice\App\Debug;
use CloudLinux\SmartAdvice\App\View;
use CloudLinux\SmartAdvice\App\Service\Analytics;

/**
 * Admin
 */
class Admin extends View {

	/**
	 * Main JS script handle.
	 *
	 * @var string
	 */
	const JS_SCRIPT_HANDLE = 'cl-smart-advice-admin';

	/**
	 * Analytics service.
	 *
	 * @var Analytics
	 */
	private $analytics;

	/**
	 * Debug.
	 *
	 * @var Debug
	 */
	private $debug;

	/**
	 * Constructor.
	 *
	 * @param Analytics $analytics Analytics service.
	 * @param Debug     $debug Debug.
	 */
	public function __construct( $analytics, $debug ) {
		$this->analytics = $analytics;
		$this->debug     = $debug;

		add_action( 'admin_enqueue_scripts', array( $this, 'scripts' ) );
		add_action( 'admin_init', array( $this, 'init_analytics_session' ) );
	}

	/**
	 * Analytics instance.
	 *
	 * @return Analytics
	 */
	public function analytics() {
		return $this->analytics;
	}

	/**
	 * Scripts and styles.
	 *
	 * @param string $hook_suffix The current admin page.
	 *
	 * @return void
	 *
	 * @phpcs:disable WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
	 */
	public function scripts( $hook_suffix ) {

		// Load styles and Sentry library only on dashboard and plugin settings page.
		$screen      = get_current_screen();
		$should_load = in_array( $screen->id, array( 'dashboard', 'settings_page_' . CL_SMART_ADVICE_SLUG ) );

		// Styles.
		wp_register_style( 'cl-smart-advice-admin', false, array(), CL_SMART_ADVICE_VERSION );

		$styles_paths = array();
		if ( $should_load ) {
			$styles_paths[] = CL_SMART_ADVICE_FOLDER_PATH . '/assets/admin.css';
		}

		if ( $this->should_hide_object_cache_banners() ) {
			$styles_paths[] = CL_SMART_ADVICE_FOLDER_PATH . '/assets/object-cache.css';
		}

		foreach ( $styles_paths as $path ) {
			if ( is_readable( $path ) ) {
				wp_add_inline_style( 'cl-smart-advice-admin', file_get_contents( $path ) );
				wp_enqueue_style( 'cl-smart-advice-admin' );
			}
		}

		// Scripts.
		$data = apply_filters(
			'cl_smart_advice_js_data',
			array(
				'analyticsApiUrl' => $this->analytics()->get_endpoint_url( Analytics::ENDPOINT_SEND ),
				'sessionData'     => $this->analytics()->get_js_session_data(),
				'sentry'          => array(
					'projectId'   => $this->debug->project_id(),
					'key'         => $this->debug->key(),
					'environment' => $this->debug->environment(),
					'release'     => $this->debug->release(),
					'tags'        => $this->debug->tags(),
					'user'        => $this->debug->userdata(),
				),
			)
		);
		wp_register_script( self::JS_SCRIPT_HANDLE, false, array( 'jquery' ), CL_SMART_ADVICE_VERSION, true );
		wp_add_inline_script( self::JS_SCRIPT_HANDLE, 'var clSmartAdviceConfig = ' . wp_json_encode( $data ), 'before' );

		$scripts_paths = array();
		if ( $should_load ) {
			$scripts_paths[] = CL_SMART_ADVICE_FOLDER_PATH . '/assets/sentry.min.js';
		}

		// The admin script has to be after Sentry library in order for it to work as expected.
		$scripts_paths[] = CL_SMART_ADVICE_FOLDER_PATH . '/assets/admin.js';

		foreach ( $scripts_paths as $path ) {
			if ( is_readable( $path ) ) {
				wp_enqueue_script( self::JS_SCRIPT_HANDLE, '', array( 'jquery' ), CL_SMART_ADVICE_VERSION, true );
				wp_add_inline_script( self::JS_SCRIPT_HANDLE, file_get_contents( $path ) );
			}
		}
	}

	/**
	 * Checks if object cache banners should be hidden.
	 *
	 * @return bool True if banners should be hidden, false otherwise.
	 */
	public function should_hide_object_cache_banners() {
		return defined( 'WP_REDIS_DISABLE_BANNERS' ) &&
				WP_REDIS_DISABLE_BANNERS &&
				is_plugin_active( 'redis-cache' ) &&
				defined( 'WP_PLUGIN_DIR' ) &&
				version_compare( '2.4.1', get_plugin_data( WP_PLUGIN_DIR . '/redis-cache/redis-cache.php' )['Version'], '>=' );
	}

	/**
	 * Initialized the analytics session using request data.
	 *
	 * @return void
	 */
	public function init_analytics_session() {
		$this->analytics()->init_session( $_GET ); // @phpcs:ignore WordPress.Security.NonceVerification
	}
}

