<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2022 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\SmartAdvice\App\Views;

use CloudLinux\SmartAdvice\App\View;
use CloudLinux\SmartAdvice\App\Advice\Manager as AdviceManager;
use CloudLinux\SmartAdvice\App\Notice\Manager as NoticeManager;

/**
 * Settings ui
 */
class Settings extends View {
	/**
	 * Advice manager.
	 *
	 * @var AdviceManager
	 */
	private $adviceManager;

	/**
	 * Notice manager.
	 *
	 * @var NoticeManager
	 */
	private $noticeManager;

	/**
	 * Constructor.
	 *
	 * @param AdviceManager $adviceManager advice.
	 * @param NoticeManager $noticeManager advice.
	 */
	public function __construct( AdviceManager $adviceManager, NoticeManager $noticeManager ) {
		$this->adviceManager = $adviceManager;
		$this->noticeManager = $noticeManager;

		if ( ! empty( $this->adviceManager->advices() ) ) {
			add_action( 'admin_init', array( $this, 'actions' ) );
			add_action( 'admin_menu', array( $this, 'menu' ) );
			add_action( 'admin_notices', array( $noticeManager, 'display' ) );
		}

		add_action( 'wp_ajax_cl_smart_advice_apply', array( $this, 'ajaxApply' ) );
		add_action( 'wp_ajax_cl_smart_advice_sync_statuses', array( $this, 'ajaxSyncStatuses' ) );
		add_action( 'wp_ajax_cl_smart_advice_rollback', array( $this, 'ajaxRollback' ) );
		add_action( 'wp_ajax_cl_smart_advice_payment_success', array( $this, 'ajaxPaymentSuccess' ) );
		add_action( 'wp_ajax_cl_smart_advice_agreement_text', array( $this, 'ajaxAgreementText' ) );
	}

	/**
	 * Advice manager.
	 *
	 * @return AdviceManager
	 */
	protected function adviceManager() {
		return $this->adviceManager;
	}

	/**
	 * Notice manager.
	 *
	 * @return NoticeManager
	 */
	protected function noticeManager() {
		return $this->noticeManager;
	}

	/**
	 * Add menu.
	 *
	 * @return void
	 */
	public function menu() {
		add_options_page(
			'SmartAdvice',
			'SmartAdvice',
			'manage_options',
			CL_SMART_ADVICE_SLUG,
			array( $this, 'view' )
		);
	}

	/**
	 * Settings view.
	 *
	 * @return void
	 */
	public function view() {
		$advices = $this->adviceManager()->advices();

		$advices_awp = array_filter(
			$advices,
			function ( $advice ) {
				return false === $advice->is_imunify();
			}
		);

		$advices_imunify = array_filter(
			$advices,
			function ( $advice ) {
				return true === $advice->is_imunify();
			}
		);

		$compatible = $this->adviceManager()->compatible();
		$this->render(
			'settings',
			array(
				'advices_awp'     => $advices_awp,
				'advices_imunify' => $advices_imunify,
				'compatible'      => $compatible,
			)
		);
	}

	/**
	 * Register actions.
	 *
	 * @return void
	 */
	public function actions() {
		if ( $this->isPluginPage() && ! empty( $this->actionName() ) ) {
			$advice_uid  = array_key_exists( 'advice_uid', $_GET ) ? sanitize_text_field( wp_unslash( $_GET['advice_uid'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$action_name = $this->actionName();

			if ( 'apply' === $action_name ) {
				$this->actionApply( (string) $advice_uid );
			} elseif ( 'apply-cdn' === $action_name ) {
				$this->actionApplyType( 'cdn' );
			} elseif ( 'rollback' === $action_name ) {
				$this->actionRollback( (string) $advice_uid );
			} elseif ( 'sync-statuses' === $action_name ) {
				$this->actionSyncStatuses();
			}

			$url = 'options-general.php?page=' . CL_SMART_ADVICE_SLUG;
			$this->redirectTo( $url );
		}
	}

	/**
	 * Redirect.
	 *
	 * @param string $url url.
	 *
	 * @return void
	 */
	protected function redirectTo( $url ) {
		wp_safe_redirect( get_admin_url( null, $url ) );
		exit;
	}

	/**
	 * Action apply.
	 *
	 * @param string $advice_uid advice.
	 *
	 * @return void
	 */
	public function actionApply( $advice_uid ) {
		do_action( 'cl_smart_advice_apply', $advice_uid, true );
	}

	/**
	 * Action apply.
	 *
	 * @param string $type advice.
	 *
	 * @return void
	 */
	public function actionApplyType( $type ) {
		do_action( 'cl_smart_advice_apply_type', $type, true );
	}

	/**
	 * Ajax apply.
	 *
	 * @return void
	 */
	public function ajaxApply() {
		$advice_uid         = array_key_exists( 'advice_uid', $_REQUEST ) ? sanitize_text_field( wp_unslash( $_REQUEST['advice_uid'] ) ) : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$agreement_accepted = array_key_exists( 'agreement_accepted', $_REQUEST ) ? (int) $_REQUEST['agreement_accepted'] : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended

		$this->noticeManager()->delete( $advice_uid );

		$agreement_type   = $this->adviceManager()->agreementType( $advice_uid );
		$subscription_url = $this->adviceManager()->subscriptionUpgradeUrl( $advice_uid );

		if ( false === $agreement_type || ( is_string( $agreement_type ) && 1 === $agreement_accepted ) ) {
			$this->actionApply( $advice_uid );
			$this->actionSyncStatuses( $advice_uid );

			$notice = $this->noticeManager()->get( $advice_uid );
			if ( $notice && $notice->isError() ) {
				$subscription_url = false;
			}

			$agreement_type = false;
		}

		echo wp_json_encode(
			array(
				'advice_uid'       => $advice_uid,
				'subscription_url' => $subscription_url,
				'agreement_type'   => $agreement_type,
			)
		);

		wp_die();
	}

	/**
	 * Action rollback.
	 *
	 * @param string $advice_uid advice.
	 * @param string $reason advice rollback reason.
	 *
	 * @return void
	 */
	public function actionRollback( $advice_uid, $reason = '' ) {
		do_action( 'cl_smart_advice_rollback', $advice_uid, true, $reason );
	}

	/**
	 * Ajax rollback.
	 *
	 * @return void
	 */
	public function ajaxRollback() {
		$advice_uid = array_key_exists( 'advice_uid', $_REQUEST ) ? sanitize_text_field( wp_unslash( $_REQUEST['advice_uid'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$reason     = array_key_exists( 'reason', $_REQUEST ) ? sanitize_text_field( wp_unslash( $_REQUEST['reason'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$this->actionRollback( $advice_uid, $reason );

		echo wp_json_encode( array() );

		wp_die();
	}

	/**
	 * Ajax payment success.
	 *
	 * @return void
	 */
	public function ajaxPaymentSuccess() {
		$advice_uid = array_key_exists( 'advice_uid', $_REQUEST ) ? sanitize_text_field( wp_unslash( $_REQUEST['advice_uid'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		do_action( 'cl_smart_advice_subscription', $advice_uid );

		echo wp_json_encode( array() );

		wp_die();
	}

	/**
	 * Action rollback.
	 *
	 * @param ?string $advice_uid id.
	 *
	 * @return void
	 */
	public function actionSyncStatuses( $advice_uid = null ) {
		do_action( 'cl_smart_advice_sync_statuses', $advice_uid );
	}

	/**
	 * Ajax sync.
	 *
	 * @return void
	 */
	public function ajaxSyncStatuses() {
		$this->actionSyncStatuses();

		$advices = array();
		array_map(
			function ( $advice ) use ( &$advices ) {
				$advices[ $advice->uid() ]                    = $advice->toArray();
				$advices[ $advice->uid() ]['status']          = $advice->statusUi();
				$advices[ $advice->uid() ]['stage_percent']   = $advice->stagePercent();
				$advices[ $advice->uid() ]['spinner_percent'] = $advice->spinnerPercent();
			},
			$this->adviceManager()->advices()
		);

		$notices = array();
		array_map(
			function ( $notice ) use ( &$notices ) {
				$notices[ $notice->id ]         = $notice->toArray();
				$notices[ $notice->id ]['html'] = $this->noticeManager()->render( $notice );
			},
			$this->noticeManager()->all()
		);

		$response = array(
			'pending' => $this->adviceManager()->hasPending(),
			'advices' => $advices,
			'notices' => $notices,
		);

		echo wp_json_encode( $response );

		wp_die();
	}

	/**
	 * Ajax agreement text.
	 *
	 * @return void
	 */
	public function ajaxAgreementText() {
		$type = array_key_exists( 'type', $_REQUEST ) ? sanitize_text_field( wp_unslash( $_REQUEST['type'] ) ) : ''; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$text = $this->adviceManager()->agreementText( $type );

		echo wp_json_encode(
			array(
				'text' => $text,
			)
		);

		wp_die();
	}
}
